const STYLE_ID = "cam-extension-injected-view-style";

const okIcon = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAoCAYAAACM/rhtAAADs0lEQVR4AeyXO0xTURjHv1MxBRYVB6OLNVHDIzF2onWx6OhgSYxuAhNtHIDZAUhkBhIJdbK6GWPQwVVwaeuEMZGSMFAS42PwtUirpsfvf+G298m9PbclDJ6cc8/r/33n13O+e24aogOe/gMGPaBAO5hYHT8aX0sn+9dHx2LF1ENLmeR+ApogkEqAgIoXU8uVcPk7kVwKVcWckDRsKVPc1zTQxorpIVJIDQHG36WT8bXUJqBIUoL8JtYKKbOwbRTUFyCOKb42ukRtcomZIlxUcwSgsZ1w8OXHE5CdRfgoV4lEkpqU+OgRDsvw7eVyT0A4YGfL7MTXr2VdI5l3kzwhXQFbDKf/EA0SIaQPWGtXQJ6YZHErdo7dmjJCCLFtGtQ7zKE363XsfWpYShqm/UsJ3BBOyzkCCkHYPSd968ba5KyTcxsgdo+FTT/ak4eP08LpCcr1LNLYiRu8hC1H+tdHr1tHbYAiREo3vtWxsQ+4+wwX7TyvDd/qukrRznNa2/jgL9K4sY+2CVB7m/jWx0Szig6H2odP27fbBFhuL1/04cS3BFDYOdRGo5c/87T6a8M4VGtvd2xfrnW4YQLkvmmS+8oZUG5wMx8fu/utkin+rYCmSbIkLOoR6JoFdEpwbC2EMJ2iFZAl7vlm1xUO7nqg3z112yYOAmdzxgMNAbLelK8diZMRstlwWKwhwCffXtGnP19hVys6ZCvgsIgJUAr5FoNu5TPD3dmadYR8dvYeAdJoi7d1zxfCKK63S/UmkQkwVA1tGSed2m6QVq0iHFk3yQQYroRXrAs59b0gVeGwVsd2x2vUejEBrkTnfvCEAiRb7eYgcJJoZZdh15vliDEqQofmUPsp+k5ulD9ocrxECjGn2WoPIbJabXiYdhDjue6FF1ybApX7rhmQQ5szdKmYpvkvT111PiZKhZ7FR1adDXBHICZ26v17SiGmnFZzBMz3Lj5nsa9YZF3gLCVlnXYPjh0BMRGutA9y7fuoWauaS/Sbpt2MXQHxNskKDbBhKyFLWKMQzbiu4QrIYARDOOC2qwOeU82ecHC8JyAEOiT/gbddAZhXKbjvOISi8O1l7wkIB3CU68uMcDCPcD/Ibpborxgs9GYGEELsyzP7AtS9FPoy2Xxv5gxAsQv6uFcNrZCHkrDNX9BuCC+T2nxDgLoVQLELfEzH+MuTlIKmJF8VxlINyXHMQQNtrk/7AOgufNdKgLp3HBO+PIWezDRDjxjLm+4H85iDRter1IEAVRZs1ObAA/4DAAD//5MfgEcAAAAGSURBVAMAKxaSYNc/FVMAAAAASUVORK5CYII=";

const failIcon = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAoCAYAAACM/rhtAAADRElEQVR4AeyWv08UQRTHv+/0D4Bai8XKKGfORKOx0KOzE0oTjZz/gJgYbSRAsJFGqLSCMyZaGeUvgNZYiJFTK1kTsdUoNobl+d4uy+7O7d7N7nJA4eRm5828H/PZNz/2Kjjg5T9g2QUqlUF2an3sVIfZOXWLnZMLyTo4wc5gXW3KQBYCZIUaGFwCbf4A8SvQ1iyIRpMVkyD4Niy28hI3UKDkAgzAqmtQKKAO+1IHbTV5oLqWF9QKUJeJj1UlU5ItsGPPZVqKr4I6VdkONas4XQEFzgF578A8bE5XuE8s28Fb8mN3CdIR0A9A3hJKZS2LgPXFu0JmAvYWLoTegewLR8w2ExAVbwI9yZyJIJCVzQVzNOynAsppHZU9NxoaWbVHjwDjd4OqspXTthFD7tJq6h5PBZT7a2Lb1b55Mgs0rgd1ZtreL7QkPArFeNsG6GevyNKeOB7FPX82kq0ldmTuK6Z5GyCIC934ZuBCfeIx0y8BKCdXT1PdNNrDftu3OwEIbNb2ECZjqr+X4goDEAll3HDv5IoTn6sS7wCUUGJfCidW0QDcF6KOk+4e4K/f0UTf1iO5pLR7gNMPAQVT0LnHJbEidwOQVyJVTunlInDxMlC7AKic0z0yJzeSAQOQvsaV+yPTcnxeA/BQQhk3tJL1E3fujJVpttGh93FdApDclZ+iLAApXuP3gOfzwAv556SyDOX+MZa3GRCWBKA/yCR/S3wp36NxLbKPy9GohcRN06gNkNwPsts5sVFNp9T+m7fRsJ7mqGcpsUtu66lp3AboG3Dltt/medy5D8w/C+rVm3k8A1vGJFJKKqBk8bXY5tuL69+BBzNBVVkCWP+Ym2nZU/9UQFWAD48AnH+pfec8D53Dm8ryyAT0TxN7Q72FFDiZg9zPmYnIBNQ38h0lQG8gu8MpQ0dANdiBJG67AlRfqMp9B/5z2o/dJUBXQPXXQPSl1QBvNcplk12wN0Lu6hC5rn4UNHzHagUYRiD3Y5PWWgM+KMP+lKst07D6kvtJbwjYllyAYVBSUMkCeKMfMjH0DpOrAomKMfi6jf4gY/oBQO5SCDCcRZdJ7sxFAZgit9VI1tW5QGe3lGFMsy0FaAbrRf/AA/4DAAD//+8sR+MAAAAGSURBVAMAiopCYFfZ77gAAAAASUVORK5CYII=';

function ensureStyle() {
    if (document.getElementById(STYLE_ID)) {
        return;
    }
    const style = document.createElement("style");
    style.id = STYLE_ID;
    style.textContent = `
        #cam-extension-injected-view {
            position: fixed;
            bottom: 24px;
            right: 24px;
            z-index: 2147483647;
            font-family: "Segoe UI", "Helvetica Neue", Arial, sans-serif;
            cursor: pointer;
            user-select: none;
        }

        .vc-precheck-button {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            padding: 5px 24px 5px 10px;
            border-radius: 28px;
            background: rgba(57,214,104,0.5);
            box-shadow: 0px 0px 10px 0px rgba(57,214,104,0.26);
            cursor: pointer;
            transition: transform 0.15s ease, box-shadow 0.15s ease;
            color: #1f3a2d;
            position: relative;
            border: 1px solid #39D668;
        }

        .vc-precheck-button:hover {
            transform: translateY(-1px);
            box-shadow: 0 12px 26px rgba(0, 0, 0, 0.2);
        }

        .vc-precheck-button.status-fail {
            background: rgba(255,33,44,0.5);
            box-shadow: 0px 0px 10px 0px rgba(0,0,0,0.08);
            border: 1px solid #FF212C;
        }

        .vc-precheck-logo {
            border-radius: 24px;
            background: radial-gradient(circle at 20% 20%, #ff9a8b, #ff6a88 60%, #ff99ac);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            color: #ffffff;
            font-size: 18px;
            letter-spacing: -0.5px;
        }

        .vc-precheck-logo img {
            border-radius: 50%;
            width: 30px;
            height: 30px;
        }

        .vc-precheck-text {
            display: flex;
            flex-direction: column;
            text-align: left;
            line-height: 1.1;
            color: #fff;
        }

        .vc-precheck-title {
            font-size: 10px;
        }

        .vc-precheck-subtitle {
            font-size: 10px;
        }

        .vc-precheck-status {
            width: 26px;
            height: 26px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #42c67b;
            font-size: 14px;
            font-weight: 700;
            box-shadow: inset 0 0 0 2px rgba(105, 240, 174, 0.2);
        }

        .vc-precheck-status-img {
            width: 21px;
            height: 21px;
            position: absolute;
            bottom: 0;
            right: 0;
        }

        .sidebar {
            position: fixed;
            top: 0;
            right: 0;
            bottom: 0;
            width: 360px;
            background: #fff;
            box-shadow: -8px 0 24px rgba(0, 0, 0, 0.1);
            transform: translateX(100%);
            transition: transform 0.3s ease;
            display: flex;
            flex-direction: column;
            z-index: 2147483647;
            font-family: "PingFang SC", "Helvetica Neue", Arial, sans-serif;
        }

        .sidebar.is-open {
            transform: translateX(0);
        }

        .sidebar-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 10px 16px 10px;
            background-color: #333;
            color: #fff;
            cursor: pointer;
        }

        .sidebar-brand {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .sidebar-logo {
            width: 32px;
            height: 32px;
            border-radius: 8px;
        }

        .sidebar-title {
            font-size: 16px;
            font-weight: 500;
            color: #fff;
        }

        .sidebar-close {
            border: none;
            background: transparent;
            font-size: 22px;
            cursor: pointer;
            color: #fff;
            line-height: 1;
        }

        .sidebar-tabs {
            display: flex;
            padding: 12px 24px 0;
            gap: 32px;
            border-bottom: 1px solid #e6e6e6;
        }

        .sidebar-tab {
            flex: 1;
            border: none;
            background: transparent;
            padding: 5px 0;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            font-size: 15px;
            font-weight: 500;
            color: #333;
            position: relative;
        }

        .sidebar-tab .tab-label {
            white-space: nowrap;
            position: relative;
        }

        .sidebar-tab::after {
            content: '';
            position: absolute;
            left: 0;
            right: 0;
            bottom: -1px;
            height: 3px;
            border-radius: 999px;
            background: transparent;
        }

        .sidebar-tab.active {
            color: #e83664;
        }

        .sidebar-tab.active::after {
            background: #e83664;
        }

        .tab-status-icon {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 18px;
            height: 18px;
            border-radius: 50%;
            font-size: 11px;
            font-weight: 700;
            color: #fff;
            position: absolute;
            top: -5px;
            right: -14px;
        }

        .tab-status-icon img {
            width: 14px;
            height: 14px;
        }

        .tab-panel {
            padding: 12px 24px 0;
        }

        .tab-panel-title {
            font-size: 16px;
            font-weight: 600;
            color: #1f1f1f;
        }

        .tab-panel-desc {
            margin-top: 4px;
            font-size: 13px;
            color: #7a7a7a;
        }
        
        .sidebar-content {
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            height: 100%;
        }

        .status-list {
            flex: 1;
            padding-bottom: 20px;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .status-item {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 16px 0;
            gap: 12px;
            border-bottom: 1px solid #E6E6E6;
        }

        .status-left {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .status-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            background: #F1F1F1;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
        }

        .status-icon img {
            width: 20px;
            height: 20px;
        }

        .status-text {
            display: flex;
            flex-direction: column;
            gap: 4px;
            width: 100%;
        }

        .status-title {
            font-size: 16px;
            color: #333;
        }

        .status-subtitle {
            font-size: 12px;
            color: #777;
        }

        .status-subtitle.fail {
            color: #FF212C;
        }

        .status-action {
            font-size: 12px;
            font-weight: 400;
        }

        .status-action-wrapper {
            position: relative;
            display: flex;
            align-items: center;
            justify-content: flex-end;
            min-width: 80px;
        }

        .status-action-wrapper.has-error .status-button {
            opacity: 0;
            pointer-events: none;
            transform: translateY(4px);
            transition: opacity 0.2s ease, transform 0.2s ease;
        }

        .status-item:hover .status-action-wrapper.has-error .status-button {
            opacity: 1;
            pointer-events: auto;
            transform: translateY(0);
        }

        .status-error-icon {
            position: absolute;
            right: 0;
            top: 50%;
            transform: translateY(-50%);
            transition: opacity 0.2s ease;
        }

        .status-item:hover .status-action-wrapper.has-error .status-error-icon {
            opacity: 0;
        }

        .status-pill {
            width: 20px;
            height: 20px;
        }

        .status-button {
            background: #fff;
            border: 1px solid #E83664;
            color: #E83664;
            border-radius: 8px;
            padding: 6px 14px;
            cursor: pointer;
            font-size: 12px;
        }

        .sidebar-footer {
            padding: 0 24px 24px;
            margin: 0 auto;
        }

        .sidebar-setup {
            width: 200px;
            border: none;
            border-radius: 10px;
            background: #E83664;
            color: #fff;
            font-size: 15px;
            font-weight: 600;
            padding: 14px 0;
            cursor: pointer;
        }

        .cam-extension-dragging {
            /*box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);*/
            transition: none;
        }

        .drag-handle {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 42px;
            cursor: pointer;
            z-index: 2147483648;
        }
    `;
    document.head.appendChild(style);
}

function ensureContainer() {
    const containerId = "cam-extension-injected-view";
    let container = document.getElementById(containerId);

    if (!container) {
        container = document.createElement("div");
        container.id = containerId;
        container.className = "lcam-control-panel";
        document.body.appendChild(container);
    }

    container.innerHTML = '';

    return container;
}

function isShow() {

}
var call_index = 0;
function postSync(type, data, from = "") {
    call_index++;
    const msgId = `call_${new Date().valueOf()}_${call_index}`;
    const message = Object.assign({
        requestId: msgId,
        type: type,
        from: "front",
    }, data || {})

    let timer, handleResponse;
    const timerPromise = new Promise((_, reject) => {
        timer = setTimeout(() => {
            if (handleResponse)
                window.removeEventListener("message", handleResponse);
            reject(new Error("request timeout"));
        }, 3000);
    });
    // 等待响应
    const run = new Promise((resolve, reject) => {
        function handle(event) {
            const { requestId, data, from, error } = event.data;
            // 匹配消息ID且来自扩展
            if (requestId === msgId) {
                clearTimeout(timer);
                window.removeEventListener("message", handle);
                if (error) {
                    reject(error);
                } else {
                    resolve(event.data);
                }
            }
        }
        handleResponse = handle;
        // 注册响应监听
        window.addEventListener("message", handle);
        window.postMessage(message, location.href);
    });

    //  raced between response and timeout
    return Promise.race([timerPromise, run]);
}
function mountApp() {
    ensureStyle();
    let state = {
        error: false,
        port: null,
        images: {
            vclogo: 'https://app.vibe-connect.net/image/cam-extension-inject/vc-logo.png',
            btnok: 'https://app.vibe-connect.net/image/cam-extension-inject/btn-ok.png',
            btnfail: 'https://app.vibe-connect.net/image/cam-extension-inject/btn-fail.png',
        },
        status: {
            // 玩具连接
            toyConnected: false,
            // 蓝牙usb
            bluetoothAdapter: false,
            mode: 'pc',
            // 通过pc连接
            connectByPc: true,
            // 通过手机连接
            connectByMobile: false,
            // 手机连接玩具
            mobileConnectApp: false,
            // video feedback
            videoFeedback: false,
            buildInObs: true,
            isExternalObs: false,
            obsStudio: false,
            obsToolsetInstalled: false,
            obsStudioInstalled: false,
        },
        sidebarVisible: false,
        activeTab: 'toy',
        dragOffset: { x: 0, y: 0 }, // 拖动偏移量
        tabs: [
            { key: 'toy', label: 'Toy Status', statusClass: 'ok', statusIcon: '✓' },
            { key: 'panel', label: 'Panel status', statusClass: 'alert', statusIcon: '!' }
        ],
        tabPanels: {
            toy: [
                {
                    title: 'Toy Connected',
                    statusKey: 'toyConnected',
                    okLabel: 'Connected',
                    failLabel: 'No toy found',
                    icon: 'https://app.vibe-connect.net/image/cam-extension-inject/toy-connected-icon.png',
                    type: 'button',
                    buttonText: 'Add Toy(s)',
                    action: 'addToy'
                },
                {
                    title: 'Connect by PC',
                    statusKey: 'connectByPc',
                    okLabel: '',
                    failLabel: 'Please restart the Vibe-Connect application',
                    icon: 'https://app.vibe-connect.net/image/cam-extension-inject/connect-by-pc-icon.png',
                }
            ],
            panel: [
                {
                    title: 'Video feedback',
                    statusKey: 'videoFeedback',
                    okLabel: '',
                    failLabel: 'Not Connected',
                    icon: 'https://app.vibe-connect.net/image/cam-extension-inject/video-feedback-icon.png',
                    type: 'button',
                    buttonText: 'Connect',
                    action: 'connectVideoFeedback'
                },
                {
                    title: 'Built-in OBS',
                    statusKey: 'buildInObs',
                    okLabel: 'Started',
                    failLabel: '',
                    icon: 'https://app.vibe-connect.net/image/cam-extension-inject/obs-icon.png',
                }
            ]
        }
    };

    function isToyTabOk() {
        let status = state.status;
        let pcPath = status.toyConnected && status.bluetoothAdapter && status.connectByPc;
        let mobilePath = status.toyConnected && status.connectByMobile && status.mobileConnectApp;
        return !!(pcPath || mobilePath);
    }

    function isPanelTabOk() {
        let status = state.status;
        let buildInObs = status.videoFeedback && status.buildInObs;
        let externalObs = status.videoFeedback && status.isExternalObs && status.obsStudio && status.obsToolsetInstalled && status.obsStudioInstalled;

        return !!(buildInObs || externalObs);
    }

    function getTabIcon(tabKey) {
        if (tabKey === 'toy') {
            return isToyTabOk() ? okIcon : failIcon;
        }
        if (tabKey === 'panel') {
            return isPanelTabOk() ? okIcon : failIcon;
        }
        return okIcon;
    }

    function cloneItem(item) {
        return Object.assign({}, item);
    }

    function buildStatusItem(baseItem, statusValue, okLabel, failLabel) {
        let processed = cloneItem(baseItem);
        processed.status = statusValue ? 'ok' : 'fail';
        if (processed.status === 'ok' && okLabel) {
            processed.displaySubtitle = okLabel;
        } else if (processed.status === 'fail' && failLabel) {
            processed.displaySubtitle = failLabel;
        } else {
            processed.displaySubtitle = '';
        }
        return processed;
    }

    function getCurrentStatusItems() {
        let items = state.tabPanels[state.activeTab] || [];
        let result = [];

        for (let i = 0; i < items.length; i++) {
            let item = items[i];
            let processedItem = cloneItem(item);

            if (processedItem.statusKey) {
                let statusValue = state.status[processedItem.statusKey];
                processedItem = buildStatusItem(
                    processedItem,
                    statusValue,
                    processedItem.okLabel,
                    processedItem.failLabel
                );
            } else {
                processedItem.status = processedItem.status || 'fail';
                processedItem.displaySubtitle = processedItem.displaySubtitle || '';
            }

            if (state.activeTab === 'toy' && processedItem.title === 'Connect by PC') {
                if (state.status.mode === 'pc') {
                    let bluetoothAdapterItem = buildStatusItem(
                        {
                            title: 'Bluetooth Adapter',
                            statusKey: 'bluetoothAdapter',
                            okLabel: 'Bluetooth is normal',
                            failLabel: 'Bluetooth Adapter Not Detected',
                            // icon: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAoCAYAAACM/rhtAAAB3UlEQVR4AeyY3VHCQBSFSWgAO8AOIlBA6MAOxArECtAKtAPHCrAD9Jm/dAAdyAOvJJ47IczuxpXD4AZmWCaXJJuz9357SDLshrUz/3jAY3+gkoNxHDc6nc5Tq9VatNvtrKJYSM0oihrmgDRAgVuv16MsywZBEDRNscPzptSs1+sjE1IDBFwfwsghyL7UURiGD6pIAwTcnXrxRMc9ta4GiAvaz7rZbK6m02ngMqQG6u4289YyAXdCOUiSZCV7l7Gvxp+ALsHY3B6Qdcqm8w7anGHbvYOsUzadd9DmDNvuHWSdsum8gzZn2HbvIOuUTXcpDtrGf3y76aD2DxpTziHizXEMjWFoDCZgYohvcS6TGJchNVAm3zBx+8iP8m8NEBOYezRrI8B5ZRvgVmmaPqsFNUBMYJaYl3YhXKqiKo6lJuC6wqDW0wDlwng8Tmaz2TXEMRztsSF91WD7bXU3UhNw5i1WKwEWRebz+Rc6vLNR9Cv2bL+trgRW5LECFoJT72lAWX3C6+Yb8euKlzkQm07akevF1NvOKUAkjHETD5CktDyGtoM35OpLTqYjBcgkOlSDhyNj+lCAk8nkE6N+ZRIyGsklDyGjpQAlEV4Dj/+1yiW5JCcTNCCTzIXm7AF/AAAA///8EMTKAAAABklEQVQDAC1c5WDvJ62oAAAAAElFTkSuQmCC',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/bluetooth-adapter-icon.png',
                            type: 'button',
                            buttonText: 'Scan',
                            action: 'scanQrCode'
                        },
                        state.status.bluetoothAdapter,
                        'Bluetooth is normal',
                        'Bluetooth Adapter Not Detected'
                    );


                    result.push(bluetoothAdapterItem);
                    result.push(processedItem);
                } else if (state.status.mode === "mobile") {
                    let mobileAppItem = buildStatusItem(
                        {
                            title: 'Mobile Connect App',
                            statusKey: 'mobileConnectApp',
                            okLabel: 'Connected',
                            failLabel: 'Please Scan the QR code',
                            // icon: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACgAAAAoCAYAAACM/rhtAAAB3UlEQVR4AeyY3VHCQBSFSWgAO8AOIlBA6MAOxArECtAKtAPHCrAD9Jm/dAAdyAOvJJ47IczuxpXD4AZmWCaXJJuz9357SDLshrUz/3jAY3+gkoNxHDc6nc5Tq9VatNvtrKJYSM0oihrmgDRAgVuv16MsywZBEDRNscPzptSs1+sjE1IDBFwfwsghyL7UURiGD6pIAwTcnXrxRMc9ta4GiAvaz7rZbK6m02ngMqQG6u4289YyAXdCOUiSZCV7l7Gvxp+ALsHY3B6Qdcqm8w7anGHbvYOsUzadd9DmDNvuHWSdsum8gzZn2HbvIOuUTXcpDtrGf3y76aD2DxpTziHizXEMjWFoDCZgYohvcS6TGJchNVAm3zBx+8iP8m8NEBOYezRrI8B5ZRvgVmmaPqsFNUBMYJaYl3YhXKqiKo6lJuC6wqDW0wDlwng8Tmaz2TXEMRztsSF91WD7bXU3UhNw5i1WKwEWRebz+Rc6vLNR9Cv2bL+trgRW5LECFoJT72lAWX3C6+Yb8euKlzkQm07akevF1NvOKUAkjHETD5CktDyGtoM35OpLTqYjBcgkOlSDhyNj+lCAk8nkE6N+ZRIyGsklDyGjpQAlEV4Dj/+1yiW5JCcTNCCTzIXm7AF/AAAA///8EMTKAAAABklEQVQDAC1c5WDvJ62oAAAAAElFTkSuQmCC',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/vc-logo.png',
                            type: 'button',
                            buttonText: 'Scan',
                            action: 'scanQrCode'
                        },
                        state.status.mobileConnectApp,
                        'Connected',
                        'Please Scan the QR code'
                    );

                    let mobileItem = buildStatusItem(
                        {
                            title: 'Connect by Mobile',
                            statusKey: 'connectByMobile',
                            okLabel: '',
                            failLabel: '',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/connect-by-mobile-icon.png',
                        },
                        state.status.connectByMobile,
                        '',
                        ''
                    );

                    result.push(mobileAppItem);
                    result.push(mobileItem);
                }
            } else if (state.activeTab === 'panel' && processedItem.title === 'Built-in OBS') {
                if (state.status.isExternalObs === false) {
                    result.push(processedItem);
                } else {
                    let obsStudioItem = buildStatusItem(
                        {
                            title: 'OBS Studio',
                            statusKey: 'obsStudio',
                            okLabel: 'Connected',
                            failLabel: 'Please open software first',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/obs-icon.png',
                            type: 'button',
                            buttonText: 'Open',
                            action: 'externalOpenObs'
                        },
                        state.status.obsStudio,
                        'Open',
                        'Please open software first'
                    );

                    let obsToolsetItem = buildStatusItem(
                        {
                            title: 'HonyPlayBox-OBS-Toolset',
                            statusKey: 'obsToolsetInstalled',
                            okLabel: '',
                            failLabel: 'Please install toolset first',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/toolset-icon.png',
                            type: 'button',
                            buttonText: 'Install',
                            action: 'externalOpenObs'
                        },
                        state.status.obsToolsetInstalled,
                        'Install',
                        'Please install toolset first'
                    );

                    let obsStudioInstalled = buildStatusItem(
                        {
                            title: 'OBS Studio Installation',
                            statusKey: 'obsStudioInstalled',
                            okLabel: '',
                            failLabel: 'Please install software first',
                            icon: 'https://app.vibe-connect.net/image/cam-extension-inject/obs-icon.png',
                            type: 'button',
                            buttonText: 'Install',
                            action: 'externalOpenObs'
                        },
                        state.status.obsStudioInstalled,
                        'Install',
                        'Please install software first'
                    );

                    result.push(obsStudioItem);
                    result.push(obsToolsetItem);
                    result.push(obsStudioInstalled);
                }
            } else {
                result.push(processedItem);
            }
        }

        return result;
    }

    function buildStatusItemsMarkup() {
        return getCurrentStatusItems().map(function (item) {
            let actionMarkup = '';

            if (item.type === 'button' && item.status !== 'ok') {
                actionMarkup = `
                    <button class="status-button status-action" data-action="${item.action || ''}">
                        ${item.buttonText || ''}
                    </button>
                    <img class="status-action status-pill status-error-icon" src="${failIcon}" />
                `;
            } else {
                actionMarkup = `
                    <img class="status-action status-pill" src="${item.status === 'ok' ? okIcon : failIcon}" />
                `;
            }

            return `
                <div class="status-item">
                    <div class="status-left">
                        <div class="status-icon">
                            <img src="${item.icon}" />
                        </div>
                        <div class="status-text">
                            <div class="status-title">${item.title}</div>
                            <div class="status-subtitle ${item.status}">
                                ${item.displaySubtitle || ''}
                            </div>
                        </div>
                    </div>
                    <div class="status-action-wrapper ${item.status !== 'ok' ? 'has-error' : ''}">
                        ${actionMarkup}
                    </div>
                </div>
            `;
        }).join('');
    }

    function buildTabsMarkup() {
        return state.tabs.map(function (tab) {
            let isActive = state.activeTab === tab.key;
            return `
                <button class="sidebar-tab ${isActive ? 'active' : ''}" data-tab="${tab.key}">
                    <div class="tab-label">
                        <span>${tab.label}</span>
                        <span class="tab-status-icon">
                            <img src="${getTabIcon(tab.key)}" />
                        </span>
                    </div>
                </button>
            `;
        }).join('');
    }

    function buildPrecheckBtn() {
        return `
            <button class="vc-precheck-button ${(isToyTabOk() && isPanelTabOk()) ? 'status-ok' : 'status-fail'}">
                <span class="vc-precheck-logo">
                    <img src="${state.images.vclogo}" />
                </span>
                <span class="vc-precheck-text">
                    <span class="vc-precheck-title">Pre-Stream</span>
                    <span class="vc-precheck-subtitle">Check</span>
                </span>
                <img class="vc-precheck-status-img" src="${(isToyTabOk() && isPanelTabOk()) ? state.images.btnok : state.images.btnfail}" />
            </button>
        `
    }

    // 限制位置在视窗内的辅助函数
    function constrainToViewport(container, transformX, transformY) {
        if (!container) return { x: 0, y: 0 };

        // 获取元素尺寸
        const containerWidth = container.offsetWidth || container.getBoundingClientRect().width;
        const containerHeight = container.offsetHeight || container.getBoundingClientRect().height;
        const viewportWidth = window.innerWidth;
        const viewportHeight = window.innerHeight;

        // 容器初始位置是 bottom: 24px, right: 24px
        // 计算初始位置（从左上角计算）
        const initialRight = 24;
        const initialBottom = 24;
        const initialX = viewportWidth - containerWidth - initialRight;
        const initialY = viewportHeight - containerHeight - initialBottom;

        // transform 是相对于初始位置的偏移
        // 计算实际位置 = 初始位置 + transform
        const actualX = initialX + transformX;
        const actualY = initialY + transformY;

        // 限制实际位置在视窗内
        const minX = 0;
        const maxX = Math.max(0, viewportWidth - containerWidth);
        const minY = 0;
        const maxY = Math.max(0, viewportHeight - containerHeight);

        const constrainedX = Math.max(minX, Math.min(maxX, actualX));
        const constrainedY = Math.max(minY, Math.min(maxY, actualY));

        // 转换回 transform 的值（相对于初始位置）
        return {
            x: constrainedX - initialX,
            y: constrainedY - initialY
        };
    }

    function render() {
        const containerId = "cam-extension-injected-view";
        let container = document.getElementById(containerId);

        // 在重新渲染前，保存当前的 transform 偏移量
        if (container && container.style.transform) {
            const transform = container.style.transform;
            const match = transform.match(/translate3d\(([^,]+)px,\s*([^,]+)px/);
            if (match) {
                state.dragOffset.x = parseFloat(match[1]) || 0;
                state.dragOffset.y = parseFloat(match[2]) || 0;
            }
        }

        if (!container) {
            container = document.createElement("div");
            container.id = containerId;
            container.className = "lcam-control-panel";
            document.body.appendChild(container);
        }

        container.innerHTML = '';

        // 只渲染可拖动的按钮部分
        container.innerHTML = `
            <div class="drag-handle"></div>
            <div class="vc-precheck-container">
                ${buildPrecheckBtn()}
            </div>
        `;

        if (state.dragOffset.x !== 0 || state.dragOffset.y !== 0) {
            setTimeout(function () {
                const constrained = constrainToViewport(container, state.dragOffset.x, state.dragOffset.y);
                container.style.transform = `translate3d(${constrained.x}px, ${constrained.y}px, 0)`;
                state.dragOffset.x = constrained.x;
                state.dragOffset.y = constrained.y;
            }, 0);
        }

        let sidebarId = "cam-extension-sidebar";
        let sidebar = document.getElementById(sidebarId);

        if (!sidebar) {
            sidebar = document.createElement("div");
            sidebar.id = sidebarId;
            document.body.appendChild(sidebar);
        }

        let sidebarClass = state.sidebarVisible ? 'sidebar is-open' : 'sidebar';
        sidebar.className = sidebarClass;
        sidebar.innerHTML = `
            <div class="sidebar-header">
                <div class="sidebar-brand">
                    <img class="sidebar-logo" src="https://app.vibe-connect.net/image/cam-extension-inject/vc-logo-1.png" />
                    <div class="sidebar-title">Vibe-Connect Pre-Stream Check</div>
                </div>
                <button class="sidebar-close" aria-label="Close">×</button>
            </div>
            <div class="sidebar-tabs">
                ${buildTabsMarkup()}
            </div>
            <div class="sidebar-content">
                <div class="tab-panel">
                    <div class="status-list">
                        ${buildStatusItemsMarkup()}
                    </div>
                </div>
                <div class="sidebar-footer">
                    <button class="sidebar-setup">Setup</button>
                </div>
            </div>
        `;

        if (state.sidebarVisible) {
            container.style.display = 'none';
        } else {
            container.style.display = '';
        }

        bindRenderedEvents(container, sidebar);
    }

    function bindRenderedEvents(container, sidebar) {
        let precheckButton = container.querySelector('.vc-precheck-button');
        if (precheckButton) {
            precheckButton.addEventListener('click', function () {
                state.sidebarVisible = true;
                render();
            });
        }

        let closeButton = sidebar.querySelector('.sidebar-close');
        if (closeButton) {
            closeButton.addEventListener('click', function () {
                state.sidebarVisible = false;
                render();
            });
        }

        let tabButtons = sidebar.querySelectorAll('.sidebar-tab');
        Array.prototype.forEach.call(tabButtons, function (button) {
            button.addEventListener('click', function () {
                let tabKey = button.getAttribute('data-tab');
                if (tabKey && tabKey !== state.activeTab) {
                    state.activeTab = tabKey;
                    render();
                }
            });
        });

        let actionButtons = sidebar.querySelectorAll('.status-button[data-action]');
        Array.prototype.forEach.call(actionButtons, function (button) {
            button.addEventListener('click', function (event) {
                let action = button.getAttribute('data-action');
                handleAction(action);
                event.stopPropagation();
            });
        });

        let setupButton = sidebar.querySelector('.sidebar-setup');
        if (setupButton) {
            setupButton.addEventListener('click', function () {
                let action = state.activeTab === 'toy' ? 'addToy' : 'connectVideoFeedback';
                handleAction(action);
            });
        }

        // 添加拖动功能，只拖动按钮容器
        addDragFunctionality(container);
    }

    function addDragFunctionality(container) {
        const dragHandle = container.querySelector('.drag-handle');
        const precheckContainer = container.querySelector('.vc-precheck-container');
        const precheckButton = container.querySelector('.vc-precheck-button');

        if (!dragHandle || !precheckContainer) return;

        const DRAG_THRESHOLD = 5;
        let isDragging = false;
        let pointerDown = false;
        let currentX;
        let currentY;
        let initialX;
        let initialY;
        let startClientX = 0;
        let startClientY = 0;
        // 从 state 中恢复之前的偏移量
        let xOffset = state.dragOffset.x || 0;
        let yOffset = state.dragOffset.y || 0;

        function getPoint(e) {
            if (e.type.startsWith('touch')) {
                return {
                    x: e.touches[0]?.clientX || 0,
                    y: e.touches[0]?.clientY || 0
                };
            }
            return {
                x: e.clientX,
                y: e.clientY
            };
        }

        function dragStart(e) {
            // 只允许通过 drag-handle 拖动
            if (e.target !== dragHandle && !dragHandle.contains(e.target)) {
                return;
            }

            const point = getPoint(e);
            initialX = point.x - xOffset;
            initialY = point.y - yOffset;
            startClientX = point.x;
            startClientY = point.y;
            isDragging = false;
            pointerDown = true;
        }

        function dragEnd() {
            if (!pointerDown) {
                return;
            }

            pointerDown = false;

            if (isDragging) {
                initialX = currentX;
                initialY = currentY;
                isDragging = false;
                container.classList.remove('cam-extension-dragging');
                // 保存当前的偏移量到 state
                state.dragOffset.x = xOffset;
                state.dragOffset.y = yOffset;
            } else if (precheckButton) {
                precheckButton.click();
            }
        }

        function drag(e) {
            if (!pointerDown) {
                return;
            }

            const point = getPoint(e);
            const deltaX = point.x - startClientX;
            const deltaY = point.y - startClientY;

            if (!isDragging) {
                if (Math.abs(deltaX) > DRAG_THRESHOLD || Math.abs(deltaY) > DRAG_THRESHOLD) {
                    isDragging = true;
                    container.classList.add('cam-extension-dragging');
                } else {
                    return;
                }
            }

            e.preventDefault();

            currentX = point.x - initialX;
            currentY = point.y - initialY;

            // 使用辅助函数限制拖动范围在视窗内
            const constrained = constrainToViewport(container, currentX, currentY);
            currentX = constrained.x;
            currentY = constrained.y;

            xOffset = currentX;
            yOffset = currentY;

            // 只对按钮容器应用 transform，而不是整个 container
            setTranslate(currentX, currentY, container);
            // 实时更新 state 中的偏移量
            state.dragOffset.x = xOffset;
            state.dragOffset.y = yOffset;
        }

        function setTranslate(xPos, yPos, el) {
            el.style.transform = `translate3d(${xPos}px, ${yPos}px, 0)`;
        }

        // 鼠标事件
        dragHandle.addEventListener("mousedown", dragStart);
        document.addEventListener("mousemove", drag);
        document.addEventListener("mouseup", dragEnd);

        // 触摸事件
        dragHandle.addEventListener("touchstart", dragStart);
        document.addEventListener("touchmove", drag);
        document.addEventListener("touchend", dragEnd);

        // 防止选择文本
        container.addEventListener("dragstart", function (e) {
            e.preventDefault();
        });

        // 窗口大小改变时，重新检查并调整位置，确保按钮在视窗内
        let resizeTimeout;
        window.addEventListener("resize", function () {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(function () {
                const constrained = constrainToViewport(container, state.dragOffset.x, state.dragOffset.y);
                if (constrained.x !== state.dragOffset.x || constrained.y !== state.dragOffset.y) {
                    setTranslate(constrained.x, constrained.y, container);
                    state.dragOffset.x = constrained.x;
                    state.dragOffset.y = constrained.y;
                }
            }, 100);
        });
    }

    async function sendMessageToSW(data) {
        try {
            await chrome.runtime.sendMessage(data);
        } catch (error) {
            console.error('Error sending message:', error);
        }
    }

    function handleAction(action) {
        if (action === 'addToy') {
            sendMessageToSW({
                type: 'Pre_Check_Add_Toys'
            });
        } else if (action === 'scanQrCode') {
            sendMessageToSW({
                type: 'Pre_Check_Sacn_Qr_Code'
            });
        } else if (action === 'connectVideoFeedback') {
            console.log('handle action connectVideoFeedback');
            sendMessageToSW({
                type: 'Pre_Check_Connect_Video_Feedback',
                url: location.href
            });
        } else if (action === 'externalOpenObs') {
            sendMessageToSW({
                type: 'Pre_Check_External_Open_Obs',
                url: location.href
            });
        }
    }

    function updateStatus(statusData) {
        if (!statusData) {
            return;
        }

        let hasChanges = false;
        for (let key in statusData) {
            if (Object.prototype.hasOwnProperty.call(state.status, key)) {
                if (state.status[key] !== statusData[key]) {
                    state.status[key] = statusData[key];
                    hasChanges = true;
                }
            }
        }

        if (hasChanges) {
            render();
        }
    }

    function onMessage(event) {
        let data = event.data;

        if (data && data.type === 'prestream-check-status') {
            console.log(data);
            const status = data.data;
            // const status = {
            //     "mode": "pc",
            //     "pingUsb": true,
            //     "usb": true,
            //     "usbOccupy": false,
            //     "os": "windows",
            //     "isWin": true,
            //     "pingObs": true,
            //     "obsRunning": false,
            //     "obsPlugin": true,
            //     "isExternalObs": false,
            // };

            // const status = JSON.parse(localStorage.getItem('test'));


            let statusData = {
                // 蓝牙usb
                bluetoothAdapter: false,
                mode: 'pc',
                // 通过pc连接
                connectByPc: true,
                // 通过手机连接
                connectByMobile: false,
                // video feedback
                videoFeedback: false,
                buildInObs: true,
                isExternalObs: false,
                obsStudio: false,
                obsToolsetInstalled: false,
                obsStudioInstalled: false,
            };
            if (status.mode === 'pc') {
                statusData.mode = 'pc';
                statusData.connectByPc = status.pingObs;
                statusData.connectByMobile = false;
                statusData.mobileConnectApp = false;

                if (status.isWin === true) {
                    statusData.bluetoothAdapter = status.usb;
                } else {
                    statusData.bluetoothAdapter = true;
                }
            } else if (status.mode === 'mobile') {
                statusData.mode = 'mobile';
                statusData.connectByPc = false;
                statusData.connectByMobile = true;
                // statusData.mobileConnectApp = status.mobileConnectApp;

                statusData.obsToolsetInstalled = status.obsPlugin;
            }

            
            if (status.isExternalObs) {
                statusData.buildInObs = false;
                statusData.isExternalObs = true;
                statusData.obsStudio = status.externalObsRunning;
                statusData.obsStudioInstalled = status.externalObsRunning;
                statusData.obsToolsetInstalled = status.obsPlugin;

                statusData.videoFeedback = status.externalObsRunning;
            } else {
                statusData.buildInObs = true;
                statusData.isExternalObs = false;

                statusData.videoFeedback = status.buildInObsRunning;
            }

            updateStatus(statusData);
        } else if (data && data.type === 'toyconnector-sync-list') {
            console.log(data);

            let toyConnected = false;
            if (Array.isArray(data.toys)) {
                for (let i = 0; i < data.toys.length; i++) {
                    if (data.toys[i].status === "connected") {
                        toyConnected = true;
                        break;
                    }
                }
            }
            let statusData = {
                toyConnected: toyConnected,
            };
            
            if (toyConnected && state.status.connectByMobile) {
                statusData.mobileConnectApp = true;
            } else {
                statusData.mobileConnectApp = false;
            }
            
            updateStatus(statusData);
        }
    }

    function initMessageListener() {
        window.addEventListener('message', onMessage);
    }

    // function initPort() {
    //     const port = chrome.runtime.connect({ name: "content-script" });

    //     if (port) {
    //         state.port = port;
    //     }
    // }

    // initPort();
    render();
    initMessageListener();
}

if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", mountApp);
} else {
    mountApp();
}